/****************************************************************************\
**
** Zoom.js
**
** Source code for handling scaling of the Silverlight content.
**
** (c) 1986 - 2009 Quark, Inc.
** All rights reserved.
**
\****************************************************************************/
/**
 *
 * OnPluginLoad
 *
 * Handles plug-in load event and adds event listeners for following events
 * OnResize, FullScreenChange & KeyUP
 *
 */
function OnPluginLoad(sender, context, source)
{
    // Add OnResize handler
    sender.content.onResize = OnResized;
    // Add FullScreenChange handler
    sender.content.onFullScreenChange = OnFullScreenChanged;
    // Add KeyUp handler
    source.addEventListener("keyUp", OnKeyUp);

    // Define a XAML element for ScaleTransform and apply it to root.
    if (source.RenderTransform == null) {
        var xamlFragment = '<TransformGroup><ScaleTransform Name="ZoomTx" ScaleX="1.0" ScaleY="1.0"/><TranslateTransform X="0" Y="0"/></TransformGroup>';
        var scaleTx = sender.content.createFromXaml(xamlFragment, false);
        source.RenderTransform = scaleTx;
    }
}

/**
 *
 * OnResized
 *
 * Handler for resize events. Updates the view when plug-in size changes.
 *
 */
function OnResized(sender, args) {
    UpdateLayout(sender, args)
}

/**
 *
 * OnFullScreenChanged
 *
 * Handler for full screen change event. Updates the view to show\exit full screen.
 *
 */
function OnFullScreenChanged(sender, args) {
    UpdateLayout(sender, args)
}

/**
 *
 * OnKeyUp
 *
 * Handler for key up events. Handles page navigation and switch to zoom commands.
 *
 */
function OnKeyUp(sender, keyEventArgs)
{
    if (keyEventArgs.shift == true && keyEventArgs.ctrl == true
        && (keyEventArgs.key == 35)) { // Ctrl+Shift+F
        SwitchToFullScreen(sender, keyEventArgs);
    }
}

/**
 *
 * SwitchToFullScreen
 *
 * Switches the view to full screen mode.
 *
 */
function SwitchToFullScreen(sender, args) {
    var plugin = sender.getHost();
    plugin.content.FullScreen = !plugin.FullScreen;
}

/**
 *
 * UpdateLayout
 *
 * Updates the view according to the current and actual size of the plug-in by
 * scaling the contents honoring the aspect ratio.
 *
 */
function UpdateLayout(sender,args) {
    var xForm = sender.RenderTransform;

    if (xForm != null) {
        var plugin = sender.getHost();

        var scaleX = plugin.content.actualWidth / sender.Width;
        var scaleY = plugin.content.actualHeight / sender.Height;

        var minScale = Math.min(scaleX, scaleY);

        // Set scaling
        xForm.Children.GetItem(0).ScaleY = minScale;
        xForm.Children.GetItem(0).ScaleX = minScale;

        // Set translation
        xForm.Children.GetItem(1).X = plugin.content.FullScreen
            ? (plugin.content.actualWidth - sender.Width * minScale)/2 : 0;
        xForm.Children.GetItem(1).Y = plugin.content.FullScreen
            ? (plugin.content.actualHeight - sender.Height * minScale)/2 : 0;
    }
}
